/************************************************************************
 * @file: AudioStreaming.h
 *
 * @version: 0.1
 *
 * @description: This header file contains Interface class Streaming.
 * Streaming class is used for 1)Logging, 2) sending playback and receving
 * capture PCM data and 3) getting the stream statistics from the Backend
 * library.The application should inherit the Streaming class for
 * implementation of stream statistics and data processing. The Application
 * should pass the streaming class handle to Backend. Streaming class has
 * default implementation for LoggingInterface class, Application can also
 * implement the Log functions for LoggingInterface class.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef _ADIT_UTILITY_AUDIOSTREAMING_H_
#define _ADIT_UTILITY_AUDIOSTREAMING_H_

#include <cstring>
#include <string>

#include "LoggingInterface.h"
#include "AudioTypes.h"

namespace adit
{

namespace utility
{

namespace audio
{

struct StreamStatistics
{
    enum ChangeFlag
    {
        XRUN_UNKNOWN   = 0x0,
        XRUN_CAPTURE   = 0x1,
        XRUN_PLAYBACK  = 0x2,
        XRUN_CAPNPLAY  = 0x3
    };

    uint64_t     time;     /**< Stream Time in frames    */
    ChangeFlag   flag;     /**< Indicates updated stat's */
    struct
    {
        uint32_t capture;  /**< xrun count for capture   */
        uint32_t playback; /**< xrun count for playback  */
    } xruns;
};

class Streaming : public LoggingInterface
{

public:

    Streaming(void) {};
    virtual ~Streaming(void) {};

    virtual void error(const std::string& data) const override
    {
        std::cerr << data << std::endl;
    };

    virtual void warning(const std::string& data) const override
    {
        std::cout << data << std::endl;
    };

    virtual void info(const std::string& data) const override
    {
        std::cout << data << std::endl;
    };

    virtual void debug(const std::string& data) const override
    {
        std::cout << data << std::endl;
    };

    virtual eLogLevel checkLogLevel(void) const override
    {
        /* Default log level is Info */
        return LL_INFO;
    };

    /**
     * This is a asynchronous callback function called from Backend which
     * provides captured PCM data from inCard to Application and/or playback PCM data from Application to outCard.
     * @param [IN]      in     Provides buffer which holds captured stream data (nullptr in case outCard was given only)
     * @param [OUT]     out    Application has to provide buffer for playback (nullptr in case inCard was given only)
     * @param [IN|OUT]  frames Number of frames read during capture or to write during playback
     * @returns OK      Normal operation streaming shall continue
     * @returns STOP    Stops streaming (see Note)
     * @returns ABORT   Aborts streaming and no further processing call will happen
     * @note If STOP error is returned and also fader is configured for specified time,
     *       Application has to provide data till the configure Fade time, because process
     *       callback will be called from Backend to get data from application till reaching
     *       the configured fade time.
     */
    virtual AudioState processing(unsigned char *in, unsigned char **out, uint32_t &frames) = 0;

    /**
     * This is a asynchronous callback function called from Backend and might be called instead of processing.
     * @param [IN] status Provides a time stamp and the amount of occurred xruns per direction
     */
    virtual void statistics(const StreamStatistics& status) = 0;

    /**
     * This is a asynchronous callback function called from Backend and is called when streaming has stopped.
     * @param [IN] error Indicates the reason why the streaming has been stopped
     * @li OK      Stopped as requested by user
     * @li FAILURE Stopped unexpectedly e.g. blocking card access
     */
    virtual void eostreaming(const AudioError error) = 0;
};

} /* namespace audio */

} /* namespace utility */

} /* namespace adit */

#endif /* _ADIT_UTILITY_AUDIOSTREAMING_H_ */
